/**
 * Mock Kalshi API Server
 *
 * Creates a mock Kalshi API server for testing purposes.
 */

import { createServer, Server } from 'http';
import { WebSocketServer } from 'ws';

/**
 * Creates a mock Kalshi API server for testing.
 *
 * @param _port - Port to listen on (used by callers when calling listen)
 * @returns Mock server instance with close method
 */
export function createMockKalshiServer(_port: number): {
  server: Server;
  wsServer: WebSocketServer;
  close: () => Promise<void>;
} {
  // Create HTTP server
  const server = createServer((req, res) => {
    // Mock HTTP responses
    if (req.url?.startsWith('/trade-api/v2/')) {
      res.writeHead(200, { 'Content-Type': 'application/json' });
      res.end(JSON.stringify({ success: true, path: req.url }));
    } else {
      res.writeHead(404);
      res.end('Not Found');
    }
  });

  // Create WebSocket server
  const wsServer = new WebSocketServer({ server });

  wsServer.on('connection', (ws) => {
    ws.on('message', (data) => {
      // Echo message back
      ws.send(data);
    });
  });

  return {
    server,
    wsServer,
    close: () => {
      return new Promise<void>((resolve) => {
        wsServer.close(() => {
          server.close(() => {
            resolve();
          });
        });
      });
    },
  };
}
