#!/usr/bin/env node
/**
 * Comprehensive Kalshi API reference fetcher.
 * Tests all sports series, markets, orderbooks, and documents response shapes.
 *
 * Usage: node fetch-samples.mjs
 */
import crypto from 'node:crypto';
import fs from 'node:fs';
import path from 'node:path';

const API_BASE = 'https://api.elections.kalshi.com';
const OUT_DIR = path.resolve(import.meta.dirname);

// --- Load credentials ---
const keyFile = fs.readFileSync(
  path.resolve(import.meta.dirname, '../../localtestapikeys.streamrift'),
  'utf-8'
);
const lines = keyFile.split('\n');
const API_KEY_ID = lines[1].trim();
const PRIVATE_KEY_PEM = lines.slice(3).join('\n').trim();

// --- Auth ---
function signRequest(method, pathStr, timestamp) {
  const message = `${timestamp}${method}${pathStr}`;
  const signer = crypto.createSign('RSA-SHA256');
  signer.update(message);
  signer.end();
  return signer.sign(
    { key: PRIVATE_KEY_PEM, padding: crypto.constants.RSA_PKCS1_PSS_PADDING, saltLength: 32 },
    'base64'
  );
}

async function kalshiFetch(method, apiPath, queryParams = {}) {
  const url = new URL(apiPath, API_BASE);
  for (const [k, v] of Object.entries(queryParams)) {
    if (v !== undefined && v !== null) url.searchParams.set(k, String(v));
  }
  const timestamp = Math.floor(Date.now() / 1000).toString();
  const pathForSig = url.pathname + url.search;
  const sig = signRequest(method, pathForSig, timestamp);

  const res = await fetch(url.toString(), {
    method,
    headers: {
      'Content-Type': 'application/json',
      'KALSHI-ACCESS-KEY': API_KEY_ID,
      'KALSHI-ACCESS-SIGNATURE': sig,
      'KALSHI-ACCESS-TIMESTAMP': timestamp,
    },
  });
  if (!res.ok) {
    const text = await res.text();
    throw new Error(`${res.status} ${res.statusText}: ${text.slice(0, 200)}`);
  }
  return res.json();
}

function save(filename, data) {
  const outPath = path.join(OUT_DIR, filename);
  fs.writeFileSync(outPath, JSON.stringify(data, null, 2));
}

function sleep(ms) { return new Promise(r => setTimeout(r, ms)); }

// =========================================================================
async function main() {
  const report = { timestamp: new Date().toISOString(), endpoints: {} };

  // -----------------------------------------------------------------------
  // 1. GET /trade-api/v2/series — full Sports category
  // -----------------------------------------------------------------------
  console.log('1. Fetching all Sports series...');
  const seriesData = await kalshiFetch('GET', '/trade-api/v2/series', { category: 'Sports' });
  save('series-sports.json', seriesData);

  const allSeries = Array.isArray(seriesData) ? seriesData : (seriesData.series || []);
  report.endpoints['GET /trade-api/v2/series'] = {
    topLevelKeys: Object.keys(seriesData),
    arrayField: Array.isArray(seriesData) ? '(root)' : 'series',
    count: allSeries.length,
    sampleKeys: allSeries[0] ? Object.keys(allSeries[0]) : [],
    note: 'Series objects use "ticker" (NOT "series_ticker")',
  };
  console.log(`  ${allSeries.length} total sports series`);
  console.log(`  Series object keys: ${Object.keys(allSeries[0] || {}).join(', ')}\n`);

  // -----------------------------------------------------------------------
  // 2. Categorize series by sport
  // -----------------------------------------------------------------------
  const sportBuckets = {};
  for (const s of allSeries) {
    const ticker = (s.ticker || '').toUpperCase();
    const title = (s.title || '').toUpperCase();
    const tags = (s.tags || []).map(t => t.toUpperCase());

    let sport = 'OTHER';
    if (ticker.startsWith('KXNBAGAME') || ticker.includes('NBA')) sport = 'NBA';
    else if (ticker.includes('NCAABB') || title.includes('COLLEGE BASKETBALL') || title.includes("MEN'S BASKETBALL TOURNAMENT") || title.includes('MARCH MADNESS')) sport = 'CBB';
    else if (ticker.includes('NFL') || title.includes('FOOTBALL') || title.includes('SUPER BOWL')) sport = 'NFL';
    else if (ticker.includes('MLB') || title.includes('BASEBALL')) sport = 'MLB';
    else if (ticker.includes('NHL') || title.includes('HOCKEY')) sport = 'NHL';
    else if (ticker.includes('MLS') || ticker.includes('EPL') || ticker.includes('LALIGA') || ticker.includes('SERIEA') || ticker.includes('BUNDESLIGA') || title.includes('SOCCER') || title.includes('FOOTBALL') && title.includes('LEAGUE')) sport = 'SOCCER';
    else if (ticker.includes('UFC') || ticker.includes('MMA') || title.includes('UFC')) sport = 'UFC';
    else if (ticker.includes('F1') || title.includes('FORMULA')) sport = 'F1';
    else if (ticker.includes('GOLF') || title.includes('GOLF') || title.includes('PGA')) sport = 'GOLF';
    else if (ticker.includes('TENNIS') || title.includes('OPEN') && (title.includes("MEN'S") || title.includes("WOMEN'S"))) sport = 'TENNIS';

    if (!sportBuckets[sport]) sportBuckets[sport] = [];
    sportBuckets[sport].push(s);
  }

  console.log('2. Series by sport:');
  for (const [sport, list] of Object.entries(sportBuckets).sort((a,b) => b[1].length - a[1].length)) {
    console.log(`  ${sport}: ${list.length} series`);
  }
  console.log();

  // -----------------------------------------------------------------------
  // 3. For key sports, fetch markets and document shape
  // -----------------------------------------------------------------------
  // Pick representative series for each major sport
  const testSeries = {};

  // CBB — find college basketball game series
  const cbbSeries = allSeries.filter(s => {
    const t = (s.ticker || '').toUpperCase();
    const title = (s.title || '').toUpperCase();
    return t.startsWith('KXNCAABB') || t.includes('CBBGAME') ||
           title.includes("MEN'S BASKETBALL TOURNAMENT") ||
           title.includes('MARCH MADNESS') ||
           title.includes('COLLEGE BASKETBALL');
  });
  if (cbbSeries.length > 0) testSeries['CBB'] = cbbSeries;

  // NBA game series
  const nbaSeries = allSeries.filter(s => (s.ticker || '').toUpperCase().startsWith('KXNBAGAME'));
  if (nbaSeries.length > 0) testSeries['NBA'] = nbaSeries;

  // NFL
  const nflSeries = allSeries.filter(s => {
    const t = (s.ticker || '').toUpperCase();
    return t.startsWith('KXNFLGAME') || t.startsWith('KXNFL') && t.includes('GAME');
  });
  if (nflSeries.length > 0) testSeries['NFL'] = nflSeries;

  // MLB
  const mlbSeries = allSeries.filter(s => (s.ticker || '').toUpperCase().startsWith('KXMLBGAME'));
  if (mlbSeries.length > 0) testSeries['MLB'] = mlbSeries;

  // NHL
  const nhlSeries = allSeries.filter(s => (s.ticker || '').toUpperCase().startsWith('KXNHLGAME'));
  if (nhlSeries.length > 0) testSeries['NHL'] = nhlSeries;

  console.log('3. Game series found per sport:');
  for (const [sport, list] of Object.entries(testSeries)) {
    console.log(`  ${sport}: ${list.length} game series — ${list.map(s=>s.ticker).join(', ')}`);
  }
  console.log();

  // -----------------------------------------------------------------------
  // 4. Fetch markets for each sport's game series
  // -----------------------------------------------------------------------
  console.log('4. Fetching markets per sport (limit=200)...');
  const sportMarkets = {};

  for (const [sport, seriesList] of Object.entries(testSeries)) {
    const allMarkets = [];
    for (const s of seriesList.slice(0, 5)) { // cap at 5 series per sport
      await sleep(200); // rate limit courtesy
      try {
        const data = await kalshiFetch('GET', '/trade-api/v2/markets', {
          series_ticker: s.ticker,
          limit: 200,
        });
        const markets = data.markets || data.results || (Array.isArray(data) ? data : []);
        allMarkets.push(...markets);

        console.log(`  ${sport}/${s.ticker}: ${markets.length} markets${data.cursor ? ' (HAS MORE — cursor present)' : ''}`);
      } catch (e) {
        console.log(`  ${sport}/${s.ticker}: ERROR — ${e.message}`);
      }
    }
    sportMarkets[sport] = allMarkets;
    save(`markets-${sport.toLowerCase()}.json`, allMarkets);
  }
  console.log();

  // -----------------------------------------------------------------------
  // 5. Document market object shape
  // -----------------------------------------------------------------------
  console.log('5. Market object shapes:');
  for (const [sport, markets] of Object.entries(sportMarkets)) {
    if (markets.length === 0) continue;
    const sample = markets[0];
    const keys = Object.keys(sample).sort();
    console.log(`\n  ${sport} (${markets.length} markets):`);
    console.log(`    Keys: ${keys.join(', ')}`);
    console.log(`    ticker field: ${sample.ticker ? 'ticker' : ''}${sample.market_ticker ? 'market_ticker' : ''}`);
    console.log(`    Sample title: "${sample.title}"`);
    console.log(`    Sample subtitle: "${sample.subtitle || ''}"`);
    console.log(`    yes_price: ${sample.yes_price}, no_price: ${sample.no_price}`);
    console.log(`    yes_bid: ${sample.yes_bid}, yes_ask: ${sample.yes_ask}`);
    console.log(`    volume: ${sample.volume}, open_interest: ${sample.open_interest}`);
    console.log(`    close_time: ${sample.close_time}, expiry_time: ${sample.expiry_time}`);
    console.log(`    event_ticker: ${sample.event_ticker}`);
    console.log(`    status: ${sample.status}, result: ${sample.result}`);
  }
  console.log();

  // -----------------------------------------------------------------------
  // 6. Test orderbook endpoint with a few active markets
  // -----------------------------------------------------------------------
  console.log('6. Testing orderbook endpoint...');
  let orderbookSample = null;
  for (const [sport, markets] of Object.entries(sportMarkets)) {
    // Find an active/open market
    const active = markets.find(m => m.status === 'active' || m.status === 'open');
    if (!active) continue;
    const ticker = active.ticker || active.market_ticker;
    await sleep(200);
    try {
      const ob = await kalshiFetch('GET', `/trade-api/v2/orderbook/${ticker}`);
      save(`orderbook-${sport.toLowerCase()}-sample.json`, ob);
      orderbookSample = ob;
      const obData = ob.orderbook || ob;
      console.log(`  ${sport} orderbook for ${ticker}:`);
      console.log(`    Top-level keys: ${Object.keys(ob).join(', ')}`);
      console.log(`    Orderbook keys: ${Object.keys(obData).join(', ')}`);
      if (obData.yes) console.log(`    yes levels: ${Array.isArray(obData.yes) ? obData.yes.length : typeof obData.yes}`);
      if (obData.no) console.log(`    no levels: ${Array.isArray(obData.no) ? obData.no.length : typeof obData.no}`);
      if (obData.yes?.[0]) console.log(`    yes[0] shape: ${JSON.stringify(obData.yes[0])}`);
      if (obData.no?.[0]) console.log(`    no[0] shape: ${JSON.stringify(obData.no[0])}`);
    } catch (e) {
      console.log(`  ${sport}/${ticker}: ERROR — ${e.message}`);
    }
  }
  console.log();

  // -----------------------------------------------------------------------
  // 7. Test single market endpoint
  // -----------------------------------------------------------------------
  console.log('7. Testing single market endpoint...');
  for (const [sport, markets] of Object.entries(sportMarkets)) {
    const active = markets.find(m => m.status === 'active' || m.status === 'open');
    if (!active) continue;
    const ticker = active.ticker || active.market_ticker;
    await sleep(200);
    try {
      const detail = await kalshiFetch('GET', `/trade-api/v2/markets/${ticker}`);
      save(`market-detail-${sport.toLowerCase()}-sample.json`, detail);
      console.log(`  ${sport}/${ticker}:`);
      console.log(`    Top-level keys: ${Object.keys(detail).join(', ')}`);
      const m = detail.market || detail;
      console.log(`    Market keys: ${Object.keys(m).sort().join(', ')}`);
      break; // one sample is enough
    } catch (e) {
      console.log(`  ${sport}/${ticker}: ERROR — ${e.message}`);
    }
  }
  console.log();

  // -----------------------------------------------------------------------
  // 8. Test events endpoint (used by some discovery flows)
  // -----------------------------------------------------------------------
  console.log('8. Testing events endpoint...');
  for (const [sport, markets] of Object.entries(sportMarkets)) {
    const m = markets.find(m => m.event_ticker);
    if (!m) continue;
    await sleep(200);
    try {
      const eventData = await kalshiFetch('GET', `/trade-api/v2/events/${m.event_ticker}`);
      save(`event-${sport.toLowerCase()}-sample.json`, eventData);
      const ev = eventData.event || eventData;
      console.log(`  ${sport} event ${m.event_ticker}:`);
      console.log(`    Top-level keys: ${Object.keys(eventData).join(', ')}`);
      console.log(`    Event keys: ${Object.keys(ev).sort().join(', ')}`);
      break;
    } catch (e) {
      console.log(`  ${sport}/${m.event_ticker}: ERROR — ${e.message}`);
    }
  }
  console.log();

  // -----------------------------------------------------------------------
  // 9. Test unfiltered markets (what dashboard gets with no series filter)
  // -----------------------------------------------------------------------
  console.log('9. Unfiltered markets (limit=200)...');
  const unfilteredData = await kalshiFetch('GET', '/trade-api/v2/markets', { limit: 200 });
  save('markets-unfiltered-200.json', unfilteredData);
  const unfilteredMarkets = unfilteredData.markets || unfilteredData.results || (Array.isArray(unfilteredData) ? unfilteredData : []);
  console.log(`  Got ${unfilteredMarkets.length} markets`);
  console.log(`  Cursor: ${unfilteredData.cursor ? 'YES' : 'NO'}`);
  console.log(`  Top-level keys: ${Object.keys(unfilteredData).join(', ')}`);

  // What sports are in unfiltered results?
  const unfilteredSports = {};
  for (const m of unfilteredMarkets) {
    const t = (m.ticker || m.market_ticker || '').toUpperCase();
    let sport = 'OTHER';
    if (t.includes('NBA')) sport = 'NBA';
    else if (t.includes('NCAABB') || t.includes('CBB')) sport = 'CBB';
    else if (t.includes('NFL')) sport = 'NFL';
    else if (t.includes('MLB')) sport = 'MLB';
    else if (t.includes('NHL')) sport = 'NHL';
    else if (t.includes('UFC')) sport = 'UFC';
    unfilteredSports[sport] = (unfilteredSports[sport] || 0) + 1;
  }
  console.log(`  Sports breakdown:`, unfilteredSports);
  console.log();

  // -----------------------------------------------------------------------
  // 10. Summary report
  // -----------------------------------------------------------------------
  report.endpoints['GET /trade-api/v2/markets'] = {
    topLevelKeys: Object.keys(unfilteredData),
    arrayField: unfilteredData.markets ? 'markets' : (unfilteredData.results ? 'results' : '(root array)'),
    hasCursor: !!unfilteredData.cursor,
    note: 'Market objects use "ticker" (NOT "market_ticker"). Cursor present when more pages available.',
  };
  report.endpoints['GET /trade-api/v2/markets/:ticker'] = {
    note: 'Returns { market: { ... } } wrapper',
  };
  report.endpoints['GET /trade-api/v2/orderbook/:ticker'] = {
    note: 'Returns { orderbook: { yes: [[price, qty], ...], no: [[price, qty], ...] } }',
  };
  report.endpoints['GET /trade-api/v2/events/:ticker'] = {
    note: 'Returns { event: { ... } } wrapper',
  };

  report.sportSeriesCounts = {};
  for (const [sport, list] of Object.entries(testSeries)) {
    report.sportSeriesCounts[sport] = { seriesCount: list.length, tickers: list.map(s => s.ticker) };
  }
  report.sportMarketCounts = {};
  for (const [sport, markets] of Object.entries(sportMarkets)) {
    const active = markets.filter(m => m.status === 'active' || m.status === 'open').length;
    report.sportMarketCounts[sport] = { total: markets.length, active };
  }

  // Key field mapping (what the code expects vs what the API returns)
  report.fieldMapping = {
    series: {
      apiField: 'ticker',
      codeExpects: 'series_ticker',
      note: 'Code maps ticker → series_ticker in getSeries(). Verify this mapping exists.',
    },
    market: {
      apiField: 'ticker',
      codeExpects: 'market_ticker',
      note: 'Code maps ticker → market_ticker in getMarkets(). Verify this mapping exists.',
    },
  };

  save('api-report.json', report);
  console.log('10. Report saved to api-report.json');
  console.log('\n=== All files saved to docs/api-reference/ ===');
}

main().catch(e => {
  console.error('Fatal:', e);
  process.exit(1);
});
