import crypto from 'node:crypto';
import fs from 'node:fs';
import path from 'node:path';

const keyFile = fs.readFileSync(path.resolve(import.meta.dirname, '../../localtestapikeys.streamrift'), 'utf-8');
const lines = keyFile.split('\n');
const API_KEY_ID = lines[1].trim();
const PEM = lines.slice(3).join('\n').trim();

function sign(method, p, ts) {
  const s = crypto.createSign('RSA-SHA256');
  s.update(ts + method + p); s.end();
  return s.sign({ key: PEM, padding: crypto.constants.RSA_PKCS1_PSS_PADDING, saltLength: 32 }, 'base64');
}

async function f(method, apiPath, q = {}) {
  const url = new URL(apiPath, 'https://api.elections.kalshi.com');
  for (const [k, v] of Object.entries(q)) {
    if (v !== undefined && v !== null) url.searchParams.set(k, String(v));
  }
  const ts = Math.floor(Date.now() / 1000).toString();
  const sig = sign(method, url.pathname + url.search, ts);
  const r = await fetch(url.toString(), {
    method,
    headers: {
      'Content-Type': 'application/json',
      'KALSHI-ACCESS-KEY': API_KEY_ID,
      'KALSHI-ACCESS-SIGNATURE': sig,
      'KALSHI-ACCESS-TIMESTAMP': ts,
    },
  });
  if (!r.ok) {
    const text = await r.text();
    throw new Error(`${r.status}: ${text.slice(0, 200)}`);
  }
  return r.json();
}

function save(name, data) {
  fs.writeFileSync(path.join(import.meta.dirname, name), JSON.stringify(data, null, 2));
}

function sleep(ms) { return new Promise(r => setTimeout(r, ms)); }

async function main() {
  // 1. NBA open markets
  console.log('=== NBA KXNBAGAME status=open limit=200 ===');
  const nbaOpen = await f('GET', '/trade-api/v2/markets', { series_ticker: 'KXNBAGAME', status: 'open', limit: 200 });
  const nbaM = nbaOpen.markets || [];
  console.log(`  Count: ${nbaM.length}, cursor: ${!!nbaOpen.cursor}`);
  if (nbaM[0]) console.log(`  First: ${nbaM[0].ticker} — "${nbaM[0].title}" status=${nbaM[0].status}`);
  if (nbaM.length > 0) save('markets-nba-open.json', nbaOpen);

  await sleep(300);

  // 2. CBB open markets — try various series tickers
  const cbbTickers = ['KXNCAABBGAME', 'KXNCAAMBGAME', 'KXNCAAWBGAME', 'KXNCAABGAME'];
  for (const t of cbbTickers) {
    console.log(`\n=== CBB ${t} status=open limit=200 ===`);
    try {
      const d = await f('GET', '/trade-api/v2/markets', { series_ticker: t, status: 'open', limit: 200 });
      const m = d.markets || [];
      console.log(`  Count: ${m.length}, cursor: ${!!d.cursor}`);
      if (m[0]) console.log(`  First: ${m[0].ticker} — "${m[0].title}"`);
      if (m.length > 0) save(`markets-cbb-${t.toLowerCase()}-open.json`, d);
    } catch (e) {
      console.log(`  ERROR: ${e.message}`);
    }
    await sleep(300);
  }

  // 3. All open markets (no series filter) — see sports breakdown
  console.log('\n=== All open markets limit=200 ===');
  const allOpen = await f('GET', '/trade-api/v2/markets', { status: 'open', limit: 200 });
  const allM = allOpen.markets || [];
  console.log(`  Count: ${allM.length}, cursor: ${!!allOpen.cursor}`);
  const cats = {};
  allM.forEach(mk => {
    const t = (mk.ticker || '').toUpperCase();
    let c = 'OTHER';
    if (t.includes('NBAGAME')) c = 'NBA';
    else if (t.includes('NCAAB') || t.includes('NCAAM') || t.includes('NCAAW')) c = 'CBB';
    else if (t.includes('NFL')) c = 'NFL';
    else if (t.includes('MLB')) c = 'MLB';
    else if (t.includes('NHL')) c = 'NHL';
    else if (t.includes('UFC')) c = 'UFC';
    else if (t.includes('SPORT')) c = 'MULTI-SPORT';
    cats[c] = (cats[c] || 0) + 1;
  });
  console.log('  Breakdown:', cats);
  save('markets-all-open-200.json', allOpen);

  await sleep(300);

  // 4. Orderbook for a live NBA market
  if (nbaM.length > 0) {
    const ticker = nbaM[0].ticker;
    console.log(`\n=== Orderbook: ${ticker} ===`);
    try {
      const ob = await f('GET', `/trade-api/v2/orderbook/${ticker}`);
      save('orderbook-nba-live.json', ob);
      const obd = ob.orderbook || ob;
      console.log(`  Keys: ${Object.keys(obd).join(', ')}`);
      console.log(`  yes levels: ${obd.yes?.length || 0}, no levels: ${obd.no?.length || 0}`);
      if (obd.yes?.[0]) console.log(`  yes[0]: ${JSON.stringify(obd.yes[0])}`);
      if (obd.no?.[0]) console.log(`  no[0]: ${JSON.stringify(obd.no[0])}`);
    } catch (e) {
      console.log(`  ERROR: ${e.message}`);
    }
  }

  await sleep(300);

  // 5. Single market detail for an open NBA market
  if (nbaM.length > 0) {
    const ticker = nbaM[0].ticker;
    console.log(`\n=== Market detail: ${ticker} ===`);
    try {
      const detail = await f('GET', `/trade-api/v2/markets/${ticker}`);
      save('market-detail-nba-live.json', detail);
      const m = detail.market || detail;
      console.log(`  Wrapper keys: ${Object.keys(detail).join(', ')}`);
      console.log(`  Market keys (${Object.keys(m).length}): ${Object.keys(m).sort().join(', ')}`);
    } catch (e) {
      console.log(`  ERROR: ${e.message}`);
    }
  }

  await sleep(300);

  // 6. Event detail for an open NBA market
  if (nbaM.length > 0 && nbaM[0].event_ticker) {
    const eticker = nbaM[0].event_ticker;
    console.log(`\n=== Event: ${eticker} ===`);
    try {
      const ed = await f('GET', `/trade-api/v2/events/${eticker}`);
      save('event-nba-live.json', ed);
      const ev = ed.event || ed;
      console.log(`  Wrapper keys: ${Object.keys(ed).join(', ')}`);
      console.log(`  Event keys: ${Object.keys(ev).sort().join(', ')}`);
      console.log(`  Title: "${ev.title}"`);
      console.log(`  series_ticker: ${ev.series_ticker}`);
      if (ed.markets) console.log(`  Nested markets count: ${ed.markets.length}`);
    } catch (e) {
      console.log(`  ERROR: ${e.message}`);
    }
  }

  // 7. Check how code's getSeries maps fields
  console.log('\n=== Field mapping verification ===');
  const seriesData = await f('GET', '/trade-api/v2/series', { category: 'Sports' });
  const firstSeries = (seriesData.series || seriesData)[0];
  console.log('  Series object has "ticker":', 'ticker' in firstSeries);
  console.log('  Series object has "series_ticker":', 'series_ticker' in firstSeries);
  console.log('  Market object has "ticker":', nbaM[0] ? 'ticker' in nbaM[0] : 'N/A');
  console.log('  Market object has "market_ticker":', nbaM[0] ? 'market_ticker' in nbaM[0] : 'N/A');

  console.log('\n=== Done ===');
}

main().catch(e => { console.error('Fatal:', e); process.exit(1); });
