/**
 * Shared types between relay and desk-ui (dashboard)
 */
export interface HttpRelayRequest {
    id: string;
    method: 'GET' | 'POST' | 'PUT' | 'DELETE' | 'PATCH';
    url: string;
    headers: Record<string, string>;
    body?: string;
}
export interface HttpRelayResponse {
    id: string;
    status: number;
    headers: Record<string, string>;
    body: string;
}
export interface WsRelayConnect {
    op: 'connect';
    id: string;
    url: string;
    headers: Record<string, string>;
}
export interface WsRelaySubscribe {
    op: 'subscribe';
    id: string;
    payload?: unknown;
}
export interface WsRelaySend {
    op: 'send';
    id: string;
    payload: unknown;
}
export interface WsRelayClose {
    op: 'close';
    id: string;
}
export interface WsRelayRegisterTelegram {
    op: 'register_telegram';
    botToken: string;
}
/** Union for dashboard (includes register_telegram). */
export type WsRelayMessage = WsRelayConnect | WsRelaySubscribe | WsRelaySend | WsRelayClose | WsRelayRegisterTelegram;
/** Union for relay app (connect | subscribe | send | close only). */
export type WsRelayOperation = WsRelayConnect | WsRelaySubscribe | WsRelaySend | WsRelayClose;
export interface WsRelayFrame {
    id: string;
    type: 'message' | 'error' | 'close';
    data?: unknown;
    error?: string;
}
export type Environment = 'prod' | 'demo';
export interface TelegramWebhookUpdate {
    update_id: number;
    message?: TelegramMessage;
    edited_message?: TelegramMessage;
    callback_query?: TelegramCallbackQuery;
}
export interface TelegramMessage {
    message_id: number;
    from?: TelegramUser;
    chat: TelegramChat;
    date: number;
    text?: string;
    entities?: TelegramMessageEntity[];
}
export interface TelegramUser {
    id: number;
    is_bot: boolean;
    first_name: string;
    username?: string;
}
export interface TelegramChat {
    id: number;
    type: 'private' | 'group' | 'supergroup' | 'channel';
    username?: string;
    first_name?: string;
}
export interface TelegramMessageEntity {
    type: string;
    offset: number;
    length: number;
}
export interface TelegramCallbackQuery {
    id: string;
    from: TelegramUser;
    message?: TelegramMessage;
    data?: string;
}
export interface TelegramWebhookFrame {
    type: 'telegram_webhook';
    update: TelegramWebhookUpdate;
    timestamp: number;
}
/** Normalized market data shape cached by the smart relay. */
export interface CachedMarket {
    ticker: string;
    venue: 'kalshi' | 'polymarket';
    /** YES price in 0..1 probability */
    price: number | null;
    yesBidCents: number | null;
    yesAskCents: number | null;
    noBidCents: number | null;
    noAskCents: number | null;
    volume: number | null;
    /** ISO-8601 */
    updatedAt: string;
}
/** Normalized orderbook shape cached by the smart relay. */
export interface CachedOrderbook {
    ticker: string;
    venue: 'kalshi' | 'polymarket';
    yes: Array<{
        priceCents: number;
        size: number;
    }>;
    no: Array<{
        priceCents: number;
        size: number;
    }>;
    /** ISO-8601 */
    updatedAt: string;
}
export interface StreamSubscribe {
    op: 'subscribe';
    channel: 'markets' | 'orderbooks';
    /** Optional ticker filter; omit = all */
    tickers?: string[];
}
export interface StreamUnsubscribe {
    op: 'unsubscribe';
    channel: 'markets' | 'orderbooks';
    tickers?: string[];
}
export type StreamClientMessage = StreamSubscribe | StreamUnsubscribe;
export interface StreamSnapshot {
    type: 'snapshot';
    channel: 'markets' | 'orderbooks';
    markets?: CachedMarket[];
    orderbooks?: CachedOrderbook[];
    timestamp: string;
}
export interface StreamUpdate {
    type: 'update';
    channel: 'markets' | 'orderbooks';
    market?: CachedMarket;
    orderbook?: CachedOrderbook;
    timestamp: string;
}
export interface StreamStatus {
    type: 'status';
    kalshiConnected: boolean;
    polymarketConnected: boolean;
    marketCount: number;
    orderbookCount: number;
    uptimeMs: number;
    timestamp: string;
}
export interface StreamError {
    type: 'error';
    message: string;
    timestamp: string;
}
export type StreamServerMessage = StreamSnapshot | StreamUpdate | StreamStatus | StreamError;
//# sourceMappingURL=relay.d.ts.map