/**
 * Path Normalization Utilities
 *
 * Utilities for normalizing paths for Kalshi API signature generation.
 *
 * Critical: The path used for signing must be normalized according to Kalshi's requirements:
 * - Strip query parameters
 * - Ensure leading slash
 * - Remove trailing slash (except for root path)
 */

/**
 * Normalizes a path for Kalshi API signature generation.
 *
 * Rules:
 * 1. Strip query parameters (everything after '?')
 * 2. Ensure leading slash
 * 3. Remove trailing slash (except for root path '/')
 *
 * @param path - The path to normalize
 * @returns Normalized path for signing
 *
 * @example
 * normalizePathForSigning('/trade-api/v2/markets?limit=10')
 * // => '/trade-api/v2/markets'
 *
 * @example
 * normalizePathForSigning('trade-api/v2/markets/')
 * // => '/trade-api/v2/markets'
 *
 * @example
 * normalizePathForSigning('/')
 * // => '/'
 */
export function normalizePathForSigning(path: string): string {
  // Step 1: Strip query parameters
  let pathWithoutQuery = path.split('?')[0];

  // Step 2: Ensure leading slash
  if (!pathWithoutQuery.startsWith('/')) {
    pathWithoutQuery = '/' + pathWithoutQuery;
  }

  // Step 3: Remove trailing slash (except for root path)
  if (pathWithoutQuery.length > 1 && pathWithoutQuery.endsWith('/')) {
    pathWithoutQuery = pathWithoutQuery.slice(0, -1);
  }

  return pathWithoutQuery;
}
