/**
 * Request ID Utilities
 *
 * Utilities for generating and validating unique request IDs.
 */

/**
 * Generates a unique request ID for relay requests.
 *
 * Format: timestamp-randomstring
 * Example: "1704067200000-a1b2c3d4e"
 *
 * Uses crypto.randomBytes for better randomness than Math.random.
 *
 * @returns Unique request ID string
 */
export function generateRequestId(): string {
  const timestamp = Date.now().toString();

  // Generate random string using crypto (more secure than Math.random)
  // Fallback to Math.random if crypto is not available
  let randomString: string;

  if (typeof crypto !== 'undefined' && crypto.getRandomValues) {
    // Browser or Node.js with crypto
    const bytes = new Uint8Array(9);
    crypto.getRandomValues(bytes);
    randomString = Array.from(bytes)
      .map((b) => b.toString(36))
      .join('')
      .substring(0, 9);
  } else {
    // Fallback for environments without crypto
    randomString = Math.random().toString(36).substring(2, 11);
  }

  return `${timestamp}-${randomString}`;
}

/**
 * Validates that a request ID is in the correct format.
 *
 * Format: timestamp-hyphen-randomstring
 * - Timestamp: numeric string
 * - Random string: alphanumeric, 5-10 characters
 *
 * @param id - Request ID to validate
 * @returns true if valid, false otherwise
 */
export function isValidRequestId(id: string): boolean {
  if (!id || typeof id !== 'string') {
    return false;
  }

  // Format: timestamp-randomstring
  const parts = id.split('-');
  if (parts.length < 2) {
    return false;
  }

  const timestamp = parts[0];
  const randomString = parts.slice(1).join('-');

  // Validate timestamp is numeric
  if (!/^\d+$/.test(timestamp)) {
    return false;
  }

  // Validate random string exists and is reasonable length
  if (!randomString || randomString.length < 5 || randomString.length > 20) {
    return false;
  }

  return true;
}
