/**
 * Discover Commands
 *
 * Explore Kalshi sports series, markets, and data availability
 */

import { Command } from 'commander';
import { createKalshiClient } from '../lib/kalshi';
import { output, outputTable, error } from '../lib/output';
import { isAuthenticated } from '../lib/config';

export const discoverCommand = new Command('discover').description(
  'Explore Kalshi sports series and markets'
);

/**
 * Ensure authenticated before running command
 */
function requireAuth(): void {
  if (!isAuthenticated()) {
    error('Not authenticated. Run `galactus auth login` first.');
    process.exit(1);
  }
}

// ============================================================================
// discover series
// ============================================================================
discoverCommand
  .command('series')
  .description('List all series (optionally filtered by category)')
  .option('--category <category>', 'Filter by category (e.g., Sports)')
  .option('--search <term>', 'Search series by title')
  .action(async (options) => {
    requireAuth();

    try {
      const client = createKalshiClient();
      let series = await client.getSeries(
        options.category ? { category: options.category } : undefined
      );

      // Filter by search term if provided
      if (options.search) {
        const term = options.search.toLowerCase();
        series = series.filter(
          (s) =>
            s.ticker.toLowerCase().includes(term) ||
            s.title?.toLowerCase().includes(term) ||
            s.category?.toLowerCase().includes(term)
        );
      }

      output(
        series.map((s) => ({
          ticker: s.ticker,
          title: s.title,
          category: s.category || '',
          subcategory: s.subcategory || '',
        }))
      );
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// discover raw (debug)
// ============================================================================
discoverCommand
  .command('raw <endpoint>')
  .description('Make a raw API request (debug)')
  .action(async (endpoint: string) => {
    requireAuth();
    try {
      const client = createKalshiClient();
      const data = await client.request('GET', endpoint.startsWith('/') ? endpoint : `/${endpoint}`);
      console.log(JSON.stringify(data, null, 2));
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// discover sports
// ============================================================================
discoverCommand
  .command('sports')
  .description('List all sports series')
  .action(async () => {
    requireAuth();

    try {
      const client = createKalshiClient();
      const series = await client.getSeries({ category: 'Sports' });

      // Group by sport type based on ticker pattern
      const grouped: Record<string, typeof series> = {};

      for (const s of series) {
        const ticker = (s.ticker || '').toUpperCase();

        // Extract sport from ticker pattern
        let sport = 'Other';
        if (ticker.includes('NBA')) sport = 'NBA';
        else if (ticker.includes('NFL')) sport = 'NFL';
        else if (ticker.includes('NHL')) sport = 'NHL';
        else if (ticker.includes('MLB')) sport = 'MLB';
        else if (ticker.includes('NCAAMB') || ticker.includes('CBB')) sport = 'CBB (Men)';
        else if (ticker.includes('NCAAWB')) sport = 'CBB (Women)';
        else if (ticker.includes('NCAAF') || ticker.includes('CFB')) sport = 'CFB';
        else if (ticker.includes('WNBA')) sport = 'WNBA';
        else if (ticker.includes('TENNIS') || ticker.includes('AO') || ticker.includes('USO') || ticker.includes('WIMB')) sport = 'Tennis';
        else if (ticker.includes('SOCCER') || ticker.includes('MLS')) sport = 'Soccer';
        else if (ticker.includes('UFC') || ticker.includes('MMA')) sport = 'MMA';

        if (!grouped[sport]) grouped[sport] = [];
        grouped[sport].push(s);
      }

      // Output grouped results
      console.log('=== SPORTS SERIES BY SPORT ===\n');

      for (const [sport, items] of Object.entries(grouped).sort()) {
        console.log(`\n${sport} (${items.length} series):`);
        console.log('─'.repeat(50));

        outputTable(
          items.map((s) => ({
            ticker: s.ticker,
            title: s.title?.slice(0, 40) || '',
          })),
          { columns: ['ticker', 'title'] }
        );
      }
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// discover markets
// ============================================================================

/** All available fields for market output */
const MARKET_FIELDS = [
  'ticker',
  'title',
  'status',
  'yes_bid',
  'yes_ask',
  'volume',
  'floor_strike',
  'cap_strike',
  'strike_type',
  'expected_expiration',
  'open_time',
] as const;

/** Default fields shown when --fields not specified */
const DEFAULT_MARKET_FIELDS = ['ticker', 'title', 'status', 'yes_bid', 'yes_ask', 'volume'];

/** Extract a field value from a market object */
function getMarketField(m: Record<string, unknown>, field: string): unknown {
  const fieldMap: Record<string, () => unknown> = {
    ticker: () => (m.ticker as string) || (m.market_ticker as string),
    title: () => m.title,
    status: () => m.status,
    yes_bid: () => m.yes_bid,
    yes_ask: () => m.yes_ask,
    volume: () => m.volume,
    floor_strike: () => m.floor_strike ?? '',
    cap_strike: () => m.cap_strike ?? '',
    strike_type: () => m.strike_type ?? '',
    expected_expiration: () => m.expected_expiration_time ?? '',
    open_time: () => m.open_time ?? '',
  };
  const getter = fieldMap[field];
  if (!getter) return '';
  return getter();
}

discoverCommand
  .command('markets <seriesTicker>')
  .description('List markets for a series')
  .option('--limit <n>', 'Limit results', '20')
  .option('--status <status>', 'Filter by status (active, closed, settled)')
  .option('--fields <fields>', `Fields to display (comma-separated). Available: ${MARKET_FIELDS.join(', ')}`)
  .action(async (seriesTicker: string, options) => {
    requireAuth();

    // Parse requested fields
    const requestedFields = options.fields
      ? options.fields.split(',').map((f: string) => f.trim())
      : DEFAULT_MARKET_FIELDS;

    // Validate fields
    const invalidFields = requestedFields.filter((f: string) => !MARKET_FIELDS.includes(f as typeof MARKET_FIELDS[number]));
    if (invalidFields.length > 0) {
      error(`Invalid fields: ${invalidFields.join(', ')}. Available: ${MARKET_FIELDS.join(', ')}`);
      process.exit(1);
    }

    try {
      const client = createKalshiClient();
      const markets = await client.getMarkets({
        series_ticker: seriesTicker,
        status: options.status,
        limit: parseInt(options.limit, 10),
      });

      output(
        markets.map((m) => {
          const result: Record<string, unknown> = {};
          for (const field of requestedFields) {
            result[field] = getMarketField(m as unknown as Record<string, unknown>, field);
          }
          return result;
        })
      );
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// discover market
// ============================================================================
discoverCommand
  .command('market <ticker>')
  .description('Get details for a single market')
  .action(async (ticker: string) => {
    requireAuth();

    try {
      const client = createKalshiClient();
      const market = await client.getMarket(ticker);
      output(market);
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// discover events
// ============================================================================
discoverCommand
  .command('events <seriesTicker>')
  .description('List events for a series')
  .option('--limit <n>', 'Limit results', '20')
  .action(async (seriesTicker: string, options) => {
    requireAuth();

    try {
      const client = createKalshiClient();
      const events = await client.getEvents({
        series_ticker: seriesTicker,
        limit: parseInt(options.limit, 10),
      });

      output(
        events.map((e) => ({
          ticker: e.event_ticker,
          title: e.title,
          subtitle: e.subtitle || '',
        }))
      );
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// discover spreads
// ============================================================================
discoverCommand
  .command('spreads')
  .description('Find all spread markets for a sport')
  .option('--sport <sport>', 'Sport: nba, nfl, nhl, cbb', 'nba')
  .option('--limit <n>', 'Limit results', '20')
  .action(async (options) => {
    requireAuth();

    const sportTickers: Record<string, string> = {
      nba: 'KXNBASPREAD',
      nfl: 'KXNFLSPREAD',
      nhl: 'KXNHLSPREAD',
      cbb: 'KXNCAAMBSPREAD',
    };

    const seriesTicker = sportTickers[options.sport.toLowerCase()];
    if (!seriesTicker) {
      error(`Unknown sport: ${options.sport}. Options: ${Object.keys(sportTickers).join(', ')}`);
      process.exit(1);
    }

    try {
      const client = createKalshiClient();
      const markets = await client.getMarkets({
        series_ticker: seriesTicker,
        status: 'active',
        limit: parseInt(options.limit, 10),
      });

      if (markets.length === 0) {
        console.log(`No spread markets found for ${options.sport.toUpperCase()}`);
        console.log(`Tried series ticker: ${seriesTicker}`);
        return;
      }

      output(
        markets.map((m) => ({
          ticker: m.market_ticker,
          title: m.title,
          floor_strike: m.floor_strike,
          cap_strike: m.cap_strike,
          yes_bid: m.yes_bid,
          yes_ask: m.yes_ask,
        }))
      );
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// discover totals
// ============================================================================
discoverCommand
  .command('totals')
  .description('Find all totals (over/under) markets for a sport')
  .option('--sport <sport>', 'Sport: nba, nfl, nhl, cbb', 'nba')
  .option('--limit <n>', 'Limit results', '20')
  .action(async (options) => {
    requireAuth();

    const sportTickers: Record<string, string> = {
      nba: 'KXNBATOTAL',
      nfl: 'KXNFLTOTAL',
      nhl: 'KXNHLTOTAL',
      cbb: 'KXNCAAMBTOTAL',
    };

    const seriesTicker = sportTickers[options.sport.toLowerCase()];
    if (!seriesTicker) {
      error(`Unknown sport: ${options.sport}. Options: ${Object.keys(sportTickers).join(', ')}`);
      process.exit(1);
    }

    try {
      const client = createKalshiClient();
      const markets = await client.getMarkets({
        series_ticker: seriesTicker,
        status: 'active',
        limit: parseInt(options.limit, 10),
      });

      if (markets.length === 0) {
        console.log(`No totals markets found for ${options.sport.toUpperCase()}`);
        console.log(`Tried series ticker: ${seriesTicker}`);
        return;
      }

      output(
        markets.map((m) => ({
          ticker: m.market_ticker,
          title: m.title,
          floor_strike: m.floor_strike,
          cap_strike: m.cap_strike,
          yes_bid: m.yes_bid,
          yes_ask: m.yes_ask,
        }))
      );
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });
