/**
 * CLI Configuration Management
 *
 * Stores credentials in ~/.config/galactus/config.json
 */

import * as fs from 'fs';
import * as path from 'path';
import * as os from 'os';

export interface CliConfig {
  accessKey?: string;
  privateKey?: string;
  environment?: 'prod' | 'demo';
  authenticatedAt?: string;
}

const CONFIG_DIR = path.join(os.homedir(), '.config', 'galactus');
const CONFIG_FILE = path.join(CONFIG_DIR, 'config.json');

/**
 * Ensure config directory exists
 */
function ensureConfigDir(): void {
  if (!fs.existsSync(CONFIG_DIR)) {
    fs.mkdirSync(CONFIG_DIR, { recursive: true, mode: 0o700 });
  }
}

/**
 * Load config from disk
 */
export function loadConfig(): CliConfig {
  try {
    if (fs.existsSync(CONFIG_FILE)) {
      const data = fs.readFileSync(CONFIG_FILE, 'utf-8');
      return JSON.parse(data);
    }
  } catch (error) {
    console.error('Warning: Failed to load config:', error);
  }
  return {};
}

/**
 * Save config to disk
 */
export function saveConfig(config: CliConfig): void {
  ensureConfigDir();
  fs.writeFileSync(CONFIG_FILE, JSON.stringify(config, null, 2), {
    mode: 0o600, // Owner read/write only
  });
}

/**
 * Set authenticated credentials
 */
export function setCredentials(
  accessKey: string,
  privateKey: string,
  environment: 'prod' | 'demo'
): void {
  const config = loadConfig();
  config.accessKey = accessKey;
  config.privateKey = privateKey;
  config.environment = environment;
  config.authenticatedAt = new Date().toISOString();
  saveConfig(config);
}

/**
 * Get credentials (returns null if not authenticated)
 */
export function getCredentials(): {
  accessKey: string;
  privateKey: string;
  environment: 'prod' | 'demo';
} | null {
  const config = loadConfig();
  if (!config.accessKey || !config.privateKey) {
    return null;
  }
  return {
    accessKey: config.accessKey,
    privateKey: config.privateKey,
    environment: config.environment || 'prod',
  };
}

/**
 * Clear credentials (logout)
 */
export function clearCredentials(): void {
  const config = loadConfig();
  delete config.accessKey;
  delete config.privateKey;
  delete config.environment;
  delete config.authenticatedAt;
  saveConfig(config);
}

/**
 * Check if authenticated
 */
export function isAuthenticated(): boolean {
  const creds = getCredentials();
  return creds !== null;
}

/**
 * Get config file path (for display)
 */
export function getConfigPath(): string {
  return CONFIG_FILE;
}
