#!/usr/bin/env bash
#
# post-receive-domino.sh — Auto-deploy domino-0 to domino.dev.g1.newbx.com
#
# Called by the post-receive router. Receives: $1=oldrev $2=newrev $3=branch
# Config: ~/g1-deploy-domino.env
# Log: ~/g1-deploy-domino.log
#
# KEY DIFFERENCES from StreamRift hook:
#   - Does NOT restart the relay (shared relay, StreamRift manages)
#   - Generates deploy manifest (JSON + markdown) for StreamRift's agents
#   - Writes deploy history (appends, doesn't overwrite)
#
# Installs to: ~/deploy-hooks/post-receive-domino.sh

set -euo pipefail

COMMIT_SHORT="${2:0:7}"
DEPLOY_BRANCH="${3:-domino-0}"

ENV_FILE="$HOME/g1-deploy-domino.env"
if [[ ! -f "$ENV_FILE" ]]; then
  echo "  ERROR: $ENV_FILE not found. Skipping deploy."
  exit 0
fi

echo "  Launching auto-deploy for $DEPLOY_BRANCH @ $COMMIT_SHORT"
echo "  Logs: ~/g1-deploy-domino.log"

nohup bash -c "
set -euo pipefail
set -a; source \"$ENV_FILE\"; set +a
export PATH=\"\$HOME/.npm-global/bin:\$PATH\"
unset GIT_DIR GIT_WORK_TREE

LOG=\"\$HOME/g1-deploy-domino.log\"
exec > \"\$LOG\" 2>&1

echo '══════════════════════════════════════════════════════'
echo '  Domino Deploy: $DEPLOY_BRANCH @ $COMMIT_SHORT'
echo \"  \$(date -u +'%Y-%m-%d %H:%M:%S UTC')\"
echo '══════════════════════════════════════════════════════'

BUILD_START=\$SECONDS

# ── Update working copy ──────────────────────────────────────────────────
if [[ ! -d \"\$BUILD_DIR\" ]]; then
  echo '▸ Cloning...'
  git clone \"\$GIT_REPO\" \"\$BUILD_DIR\" --branch \"$DEPLOY_BRANCH\"
else
  echo '▸ Updating...'
  cd \"\$BUILD_DIR\"
  git fetch origin
  git checkout \"$DEPLOY_BRANCH\"
  git reset --hard \"origin/$DEPLOY_BRANCH\"
fi
cd \"\$BUILD_DIR\"

# ── Capture git info for manifest ────────────────────────────────────────
DEPLOY_VERSION=\$(git rev-parse --short HEAD)
DEPLOY_TIME=\$(date -u +'%Y-%m-%dT%H:%M:%SZ')
DEPLOY_MSG=\$(git log -1 --format='%s')
DEPLOY_BODY=\$(git log -1 --format='%b')
DEPLOY_AUTHOR=\$(git log -1 --format='%an')
FILES_CHANGED_LIST=\$(git diff --name-only HEAD~1 HEAD 2>/dev/null || echo 'initial')
FILES_CHANGED=\$(echo \"\$FILES_CHANGED_LIST\" | wc -l | tr -d ' ')
GIT_LOG_RECENT=\$(git log --oneline -10)

# ── Build ────────────────────────────────────────────────────────────────
echo '▸ Installing dependencies...'
npm install

echo '▸ Building shared...'
npm run build --workspace=@galactus/shared

echo '▸ Building dashboard...'
VITE_DASHBOARD_PASSWORD_HASH=\"\$STAGING_DASHBOARD_PASSWORD_HASH\" \\
VITE_RELAY_HTTP_URL=/relay/http \\
VITE_RELAY_WS_URL=\"wss://\$STAGING_DOMAIN/relay/ws\" \\
VITE_ODDS_API_KEY=\"\$STAGING_ODDS_API_KEY\" \\
  npm run dashboard:build

BUILD_STATUS='success'
BUILD_DURATION=\$((\$SECONDS - \$BUILD_START))

# ── Deploy dashboard ─────────────────────────────────────────────────────
echo '▸ Deploying dashboard...'
mkdir -p \"\$DASHBOARD_DIR\"
find \"\$DASHBOARD_DIR\" -mindepth 1 -not -name '.htaccess' -not -name 'status.html' -not -name 'status.json' -not -name 'deploy-manifest.json' -not -name 'deploy-history.json' -delete 2>/dev/null || true
cp -r \"\$BUILD_DIR/apps/dashboard/dist/.\" \"\$DASHBOARD_DIR/\"

# version.json
printf '{\"version\":\"%s\",\"deployedAt\":\"%s\",\"branch\":\"%s\"}' \\
  \"\$DEPLOY_VERSION\" \"\$DEPLOY_TIME\" \"$DEPLOY_BRANCH\" > \"\$DASHBOARD_DIR/version.json\"

# .htaccess
cat > \"\$DASHBOARD_DIR/.htaccess\" << 'HTEOF'
Options -Indexes
DirectoryIndex index.html

<IfModule mod_headers.c>
  <FilesMatch \"\.(js|css|woff2?|ttf|eot|svg|png|jpg|gif|ico|webp)\$\">
    Header set Cache-Control \"public, max-age=31536000, immutable\"
  </FilesMatch>
  <FilesMatch \"\.(html|json)\$\">
    Header set Cache-Control \"no-cache, no-store, must-revalidate\"
    Header set Pragma \"no-cache\"
    Header set Expires \"0\"
  </FilesMatch>
</IfModule>

<IfModule mod_rewrite.c>
  RewriteEngine On
  RewriteBase /
  RewriteRule ^relay/(.*)\$ http://127.0.0.1:\$STAGING_RELAY_PORT/relay/\$1 [P,L]
  RewriteCond %{REQUEST_FILENAME} !-f
  RewriteCond %{REQUEST_FILENAME} !-d
  RewriteRule . /index.html [L]
</IfModule>
HTEOF

# ── NOTE: No relay deploy/restart for Domino ─────────────────────────────
echo '▸ Skipping relay deploy (shared relay managed by StreamRift)'

# ── Health check (relay, read-only) ──────────────────────────────────────
RELAY_STATUS='healthy'
RELAY_HEALTH=\$(curl -sf \"http://localhost:\$STAGING_RELAY_PORT/health\" 2>/dev/null || echo 'FAIL')
if [[ \"\$RELAY_HEALTH\" == 'FAIL' ]]; then
  echo '  ⚠ Relay health check FAILED (relay managed by StreamRift)'
  RELAY_STATUS='failed'
else
  echo '  ✓ Relay healthy'
fi

# ── Generate deploy manifest ─────────────────────────────────────────────
echo '▸ Generating deploy manifest...'

cat > \"\$DASHBOARD_DIR/deploy-manifest.json\" << MANIFEST
{
  \"branch\": \"$DEPLOY_BRANCH\",
  \"commit\": \"\$DEPLOY_VERSION\",
  \"commitMessage\": \"\$(echo \$DEPLOY_MSG | sed 's/\"/\\\\\"/g')\",
  \"commitBody\": \"\$(echo \$DEPLOY_BODY | tr '\\n' ' ' | sed 's/\"/\\\\\"/g')\",
  \"author\": \"\$DEPLOY_AUTHOR\",
  \"deployedAt\": \"\$DEPLOY_TIME\",
  \"buildDurationSec\": \$BUILD_DURATION,
  \"buildStatus\": \"\$BUILD_STATUS\",
  \"filesChanged\": \$FILES_CHANGED,
  \"filesChangedList\": \"\$(echo \$FILES_CHANGED_LIST | tr '\\n' ', ')\",
  \"relayHealthy\": \$([ \"\$RELAY_STATUS\" = 'healthy' ] && echo true || echo false),
  \"developer\": \"domino\"
}
MANIFEST

# ── Generate status.json ─────────────────────────────────────────────────
cat > \"\$DASHBOARD_DIR/status.json\" << STATUSJSON
{
  \"branch\": \"$DEPLOY_BRANCH\",
  \"commit\": \"\$DEPLOY_VERSION\",
  \"commitMessage\": \"\$(echo \$DEPLOY_MSG | sed 's/\"/\\\\\"/g')\",
  \"deployedAt\": \"\$DEPLOY_TIME\",
  \"buildDurationSec\": \$BUILD_DURATION,
  \"status\": \"\$BUILD_STATUS\",
  \"filesChanged\": \$FILES_CHANGED,
  \"relayHealthy\": \$([ \"\$RELAY_STATUS\" = 'healthy' ] && echo true || echo false),
  \"developer\": \"domino\"
}
STATUSJSON

# ── Append to deploy history ─────────────────────────────────────────────
HISTORY_FILE=\"\$DASHBOARD_DIR/deploy-history.json\"
ENTRY='{\"commit\":\"\$DEPLOY_VERSION\",\"message\":\"\$(echo \$DEPLOY_MSG | sed 's/\"/\\\\\"/g')\",\"at\":\"\$DEPLOY_TIME\",\"status\":\"\$BUILD_STATUS\",\"files\":\$FILES_CHANGED}'
if [[ -f \"\$HISTORY_FILE\" ]]; then
  # Prepend new entry, keep last 50
  python3 -c \"
import json,sys
try:
  h=json.load(open('\$HISTORY_FILE'))
except: h=[]
h.insert(0,json.loads('\$ENTRY'))
json.dump(h[:50],open('\$HISTORY_FILE','w'),indent=2)
\" 2>/dev/null || echo '[\$ENTRY]' > \"\$HISTORY_FILE\"
else
  echo '[\$ENTRY]' > \"\$HISTORY_FILE\"
fi

# ── Generate status.html (same template as StreamRift) ────────────────────
cat > \"\$DASHBOARD_DIR/status.html\" << 'STATUSHTML'
<!DOCTYPE html>
<html lang=\"en\">
<head>
<meta charset=\"UTF-8\">
<meta name=\"viewport\" content=\"width=device-width, initial-scale=1.0\">
<title>Deploy Status — Domino</title>
<style>
*{margin:0;padding:0;box-sizing:border-box}
body{background:#0a0a0f;color:#e0e0e0;font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',sans-serif;padding:2rem;min-height:100vh}
.container{max-width:700px;margin:0 auto}
h1{font-size:1.1rem;color:#888;text-transform:uppercase;letter-spacing:0.15em;margin-bottom:2rem}
.status-grid{display:grid;gap:0.75rem;margin-bottom:2rem}
.status-row{display:flex;align-items:center;gap:1rem;padding:0.75rem 1rem;background:#12121a;border-radius:8px;border:1px solid #1e1e2e}
.dot{width:12px;height:12px;border-radius:50%;flex-shrink:0}
.dot.green{background:#22c55e;box-shadow:0 0 8px #22c55e40}
.dot.red{background:#ef4444;box-shadow:0 0 8px #ef444440}
.dot.yellow{background:#eab308;box-shadow:0 0 8px #eab30840}
.dot.grey{background:#555}
.bar{flex:1;height:8px;background:#1e1e2e;border-radius:4px;overflow:hidden}
.bar-fill{height:100%;border-radius:4px;transition:width 0.5s}
.bar-fill.green{background:linear-gradient(90deg,#22c55e,#16a34a)}
.bar-fill.red{background:linear-gradient(90deg,#ef4444,#dc2626)}
.label{width:80px;font-size:0.85rem;color:#999;text-transform:uppercase;letter-spacing:0.05em}
.pipeline{display:flex;align-items:center;gap:0.5rem;margin:2rem 0;justify-content:center;flex-wrap:wrap}
.node{padding:0.5rem 1rem;background:#12121a;border:1px solid #1e1e2e;border-radius:6px;font-size:0.8rem;color:#999;position:relative}
.node.active{border-color:#22c55e;color:#22c55e}
.arrow{color:#333;font-size:1.2rem}
.card{background:#12121a;border:1px solid #1e1e2e;border-radius:8px;padding:1.25rem;margin-bottom:1.5rem}
.card-row{display:flex;justify-content:space-between;padding:0.35rem 0;border-bottom:1px solid #1a1a2a}
.card-row:last-child{border:none}
.card-label{color:#666;font-size:0.85rem}
.card-value{color:#e0e0e0;font-size:0.85rem;font-family:'SF Mono',Monaco,monospace}
.deploys{list-style:none}
.deploys li{display:flex;align-items:center;gap:0.75rem;padding:0.5rem 0;border-bottom:1px solid #1a1a2a;font-size:0.85rem}
.deploys li:last-child{border:none}
.hash{font-family:'SF Mono',Monaco,monospace;color:#7c3aed;font-size:0.8rem}
.time{color:#666;min-width:50px}
.msg{color:#ccc;flex:1}
.refresh{text-align:center;margin-top:2rem}
.refresh a{color:#555;font-size:0.75rem;text-decoration:none}
.refresh a:hover{color:#888}
</style>
</head>
<body>
<div class=\"container\">
<h1>Deploy Status — Domino</h1>
<div id=\"app\">Loading...</div>
</div>
<script>
async function load(){
  try{
    const r=await fetch('/status.json?t='+Date.now());
    const d=await r.json();
    const ok=d.status==='success';
    const rc=d.relayHealthy?'green':'red';
    let historyHtml='';
    try{
      const hr=await fetch('/deploy-history.json?t='+Date.now());
      const hist=await hr.json();
      historyHtml='<h2 style=\"font-size:0.9rem;color:#666;margin:1.5rem 0 0.75rem;text-transform:uppercase;letter-spacing:0.1em\">Recent Deploys</h2><div class=\"card\"><ul class=\"deploys\">'+
        hist.slice(0,10).map(h=>'<li><span class=\"dot '+(h.status==='success'?'green':'red')+'\" style=\"width:8px;height:8px\"></span><span class=\"hash\">'+h.commit+'</span><span class=\"time\">'+new Date(h.at).toLocaleTimeString()+'</span><span class=\"msg\">'+h.message+'</span></li>').join('')+
        '</ul></div>';
    }catch(e){}
    document.getElementById('app').innerHTML=
    '<div class=\"status-grid\">'+
      '<div class=\"status-row\"><span class=\"label\">Build</span><div class=\"bar\"><div class=\"bar-fill '+(ok?'green':'red')+'\" style=\"width:100%\"></div></div><span class=\"dot '+(ok?'green':'red')+'\"></span></div>'+
      '<div class=\"status-row\"><span class=\"label\">Deploy</span><div class=\"bar\"><div class=\"bar-fill '+(ok?'green':'red')+'\" style=\"width:100%\"></div></div><span class=\"dot '+(ok?'green':'red')+'\"></span></div>'+
      '<div class=\"status-row\"><span class=\"label\">Relay</span><div class=\"bar\"><div class=\"bar-fill '+rc+'\" style=\"width:100%\"></div></div><span class=\"dot '+rc+'\"></span></div>'+
    '</div>'+
    '<div class=\"pipeline\">'+
      '<div class=\"node '+(d.developer==='domino'?'active':'')+'\">domino.dev</div>'+
      '<span class=\"arrow\">→</span>'+
      '<div class=\"node '+(d.developer==='streamrift'?'active':'')+'\">streamrift.dev</div>'+
      '<span class=\"arrow\">→</span>'+
      '<div class=\"node '+(d.branch==='dev'?'active':'')+'\">dev.g1</div>'+
      '<span class=\"arrow\">→</span>'+
      '<div class=\"node\">g1</div>'+
      '<span class=\"arrow\">→</span>'+
      '<div class=\"node\">prod</div>'+
    '</div>'+
    '<div class=\"card\">'+
      '<div class=\"card-row\"><span class=\"card-label\">Commit</span><span class=\"card-value\">'+d.commit+'</span></div>'+
      '<div class=\"card-row\"><span class=\"card-label\">Branch</span><span class=\"card-value\">'+d.branch+'</span></div>'+
      '<div class=\"card-row\"><span class=\"card-label\">Message</span><span class=\"card-value\">'+d.commitMessage+'</span></div>'+
      '<div class=\"card-row\"><span class=\"card-label\">Deployed</span><span class=\"card-value\">'+new Date(d.deployedAt).toLocaleString()+'</span></div>'+
      '<div class=\"card-row\"><span class=\"card-label\">Build Time</span><span class=\"card-value\">'+d.buildDurationSec+'s</span></div>'+
      '<div class=\"card-row\"><span class=\"card-label\">Files Changed</span><span class=\"card-value\">'+d.filesChanged+'</span></div>'+
    '</div>'+historyHtml+
    '<div class=\"refresh\"><a href=\"javascript:load()\">↻ Refresh</a> · <a href=\"/deploy-manifest.json\">Raw Manifest</a></div>';
  }catch(e){
    document.getElementById('app').innerHTML='<p style=\"color:#ef4444\">Failed to load status. Deploy may be in progress.</p>';
  }
}
load();
setInterval(load,15000);
</script>
</body>
</html>
STATUSHTML

echo ''
echo '══════════════════════════════════════════════════════'
echo \"  Deploy complete: $DEPLOY_BRANCH @ \$DEPLOY_VERSION\"
echo \"  https://\$STAGING_DOMAIN/\"
echo \"  Status: https://\$STAGING_DOMAIN/status.html\"
echo '══════════════════════════════════════════════════════'
" </dev/null >> "$HOME/g1-deploy-domino.log" 2>&1 &

echo "  Build started in background (PID $!)."
