#!/usr/bin/env bash
#
# post-receive — Auto-deploy streamrift-0 branch to staging
# This file gets installed to ~/git/g1.1.git/hooks/post-receive on the server.
#
# The actual build+deploy runs as a detached background process so that
# `git push` returns immediately.  Progress is logged to ~/g1-deploy.log.

set -euo pipefail

# Only deploy for the streamrift-0 branch
DEPLOY_BRANCH="streamrift-0"
SHOULD_DEPLOY=false

while read -r oldrev newrev refname; do
  branch="${refname#refs/heads/}"
  if [[ "$branch" == "$DEPLOY_BRANCH" ]]; then
    SHOULD_DEPLOY=true
    COMMIT_SHORT="${newrev:0:7}"
  fi
done

if [[ "$SHOULD_DEPLOY" != "true" ]]; then
  echo "post-receive: Not $DEPLOY_BRANCH — skipping auto-deploy."
  exit 0
fi

# Load config
ENV_FILE="$HOME/g1-deploy.env"
if [[ ! -f "$ENV_FILE" ]]; then
  echo "post-receive: ERROR — $ENV_FILE not found. Skipping deploy."
  exit 0
fi

echo "post-receive: Launching auto-deploy for $DEPLOY_BRANCH @ $COMMIT_SHORT"
echo "  Logs: ~/g1-deploy.log"

# Launch the build+deploy in a fully detached subshell.
nohup bash -c "
set -euo pipefail
set -a; source \"$ENV_FILE\"; set +a
export PATH=\"\$HOME/.npm-global/bin:\$PATH\"
unset GIT_DIR GIT_WORK_TREE  # post-receive sets these to the bare repo; must clear for working copy ops

LOG=\"\$HOME/g1-deploy.log\"
echo '══════════════════════════════════════════════════════' > \"\$LOG\"
echo '  Auto-deploy: $DEPLOY_BRANCH @ $COMMIT_SHORT'      >> \"\$LOG\"
echo \"  \$(date -u +'%Y-%m-%d %H:%M:%S UTC')\"           >> \"\$LOG\"
echo '══════════════════════════════════════════════════════' >> \"\$LOG\"

# Update working copy
if [[ ! -d \"\$BUILD_DIR\" ]]; then
  echo '▸ Cloning...' >> \"\$LOG\"
  git clone \"\$GIT_REPO\" \"\$BUILD_DIR\" --branch \"$DEPLOY_BRANCH\" >> \"\$LOG\" 2>&1
else
  echo '▸ Updating...' >> \"\$LOG\"
  cd \"\$BUILD_DIR\"
  git fetch origin >> \"\$LOG\" 2>&1
  git checkout \"$DEPLOY_BRANCH\" >> \"\$LOG\" 2>&1
  git reset --hard \"origin/$DEPLOY_BRANCH\" >> \"\$LOG\" 2>&1
fi
cd \"\$BUILD_DIR\"

echo '▸ Installing dependencies...' >> \"\$LOG\"
npm install >> \"\$LOG\" 2>&1

echo '▸ Building shared...' >> \"\$LOG\"
npm run build --workspace=@galactus/shared >> \"\$LOG\" 2>&1

echo '▸ Building dashboard...' >> \"\$LOG\"
VITE_DASHBOARD_PASSWORD_HASH=\"\$STAGING_DASHBOARD_PASSWORD_HASH\" \\
VITE_RELAY_HTTP_URL=/relay/http \\
VITE_RELAY_WS_URL=\"wss://\$STAGING_DOMAIN/relay/ws\" \\
VITE_ODDS_API_KEY=\"\$STAGING_ODDS_API_KEY\" \\
  npm run dashboard:build >> \"\$LOG\" 2>&1

echo '▸ Building relay...' >> \"\$LOG\"
npm run relay:build >> \"\$LOG\" 2>&1

# Deploy dashboard
echo '▸ Deploying dashboard...' >> \"\$LOG\"
mkdir -p \"\$DASHBOARD_DIR\"
find \"\$DASHBOARD_DIR\" -mindepth 1 -not -name '.htaccess' -delete 2>/dev/null || true
cp -r \"\$BUILD_DIR/apps/dashboard/dist/.\" \"\$DASHBOARD_DIR/\"

DEPLOY_VERSION=\$(git -C \"\$BUILD_DIR\" rev-parse --short HEAD 2>/dev/null || echo '$COMMIT_SHORT')
DEPLOY_TIME=\$(date -u +'%Y-%m-%dT%H:%M:%SZ')
printf '{\"version\":\"%s\",\"deployedAt\":\"%s\"}' \"\$DEPLOY_VERSION\" \"\$DEPLOY_TIME\" > \"\$DASHBOARD_DIR/version.json\"

cat > \"\$DASHBOARD_DIR/.htaccess\" << 'HTEOF'
Options -Indexes
DirectoryIndex index.html

<IfModule mod_headers.c>
  <FilesMatch \"\\.(js|css|woff2?|ttf|eot|svg|png|jpg|gif|ico|webp)\$\">
    Header set Cache-Control \"public, max-age=31536000, immutable\"
  </FilesMatch>
  <FilesMatch \"\\.(html|json)\$\">
    Header set Cache-Control \"no-cache, no-store, must-revalidate\"
    Header set Pragma \"no-cache\"
    Header set Expires \"0\"
  </FilesMatch>
</IfModule>

<IfModule mod_rewrite.c>
  RewriteEngine On
  RewriteBase /
  RewriteRule ^relay/(.*)\$ http://127.0.0.1:8789/relay/\$1 [P,L]
  RewriteCond %{REQUEST_FILENAME} !-f
  RewriteCond %{REQUEST_FILENAME} !-d
  RewriteRule . /index.html [L]
</IfModule>
HTEOF

# Deploy relay
echo '▸ Deploying relay...' >> \"\$LOG\"
mkdir -p \"\$RELAY_DIR/packages/shared\"
cp \"\$BUILD_DIR/apps/relay/package.json\" \"\$RELAY_DIR/\"
cp \"\$BUILD_DIR/apps/relay/ecosystem.config.cjs\" \"\$RELAY_DIR/\"
cp -r \"\$BUILD_DIR/apps/relay/dist/.\" \"\$RELAY_DIR/dist/\"
cp \"\$BUILD_DIR/packages/shared/package.json\" \"\$RELAY_DIR/packages/shared/\"
cp -r \"\$BUILD_DIR/packages/shared/dist/.\" \"\$RELAY_DIR/packages/shared/dist/\"

cd \"\$RELAY_DIR\"
sed -i 's|\"@galactus/shared\": \"\\*\"|\"@galactus/shared\": \"file:./packages/shared\"|' package.json

cat > \"\$RELAY_DIR/.env\" << ENVEOF
PORT=\$STAGING_RELAY_PORT
NODE_ENV=production
CORS_ORIGIN=https://\$STAGING_DOMAIN
ENVEOF

echo '▸ Installing relay deps...' >> \"\$LOG\"
npm install --omit=dev >> \"\$LOG\" 2>&1

echo '▸ Restarting relay...' >> \"\$LOG\"
pm2 delete galactus-relay 2>/dev/null || true
pm2 start ecosystem.config.cjs >> \"\$LOG\" 2>&1
pm2 save >> \"\$LOG\" 2>&1

sleep 2
RELAY_HEALTH=\$(curl -sf \"http://localhost:\$STAGING_RELAY_PORT/health\" 2>/dev/null || echo 'FAIL')
if [[ \"\$RELAY_HEALTH\" == 'FAIL' ]]; then
  echo '  ✗ Relay health check FAILED' >> \"\$LOG\"
else
  echo '  ✓ Relay healthy' >> \"\$LOG\"
fi

echo '' >> \"\$LOG\"
echo '══════════════════════════════════════════════════════' >> \"\$LOG\"
echo \"  Deploy complete: $DEPLOY_BRANCH @ \$DEPLOY_VERSION\" >> \"\$LOG\"
echo \"  https://\$STAGING_DOMAIN/\" >> \"\$LOG\"
echo '══════════════════════════════════════════════════════' >> \"\$LOG\"
" </dev/null >> "$HOME/g1-deploy.log" 2>&1 &

echo "post-receive: Build started in background (PID $!)."
