#!/usr/bin/env bash
#
# pre-receive — Branch protection based on SSH key identity.
#
# Install to: ~/git/g1.1.git/hooks/pre-receive
#
# How it works:
#   The authorized_keys entry for each user sets GALACTUS_USER via env:
#
#   # StreamRift (full access)
#   command="env GALACTUS_USER=streamrift git-shell -c \"$SSH_ORIGINAL_COMMAND\"",no-port-forwarding,no-X11-forwarding,no-agent-forwarding ssh-ed25519 AAAA... streamrift
#
#   # Domino (domino-0 only)
#   command="env GALACTUS_USER=domino git-shell -c \"$SSH_ORIGINAL_COMMAND\"",no-port-forwarding,no-X11-forwarding,no-agent-forwarding ssh-ed25519 AAAA... domino
#
# If GALACTUS_USER is unset (e.g., local server operations), all branches are allowed.

set -euo pipefail

GALACTUS_USER="${GALACTUS_USER:-streamrift}"

# StreamRift has full access to all branches
if [[ "$GALACTUS_USER" == "streamrift" ]]; then
  exit 0
fi

# For all other users, enforce branch restrictions
# Convention: user "domino" can only push to "domino-0"
ALLOWED_BRANCH="${GALACTUS_USER}-0"

while read -r oldrev newrev refname; do
  # Allow tag pushes from anyone
  if [[ "$refname" == refs/tags/* ]]; then
    continue
  fi

  branch="${refname#refs/heads/}"

  if [[ "$branch" != "$ALLOWED_BRANCH" ]]; then
    echo ""
    echo "══════════════════════════════════════════════════════"
    echo "  ACCESS DENIED"
    echo ""
    echo "  User '$GALACTUS_USER' can only push to '$ALLOWED_BRANCH'."
    echo "  You tried to push to: '$branch'"
    echo ""
    echo "  If you need access to other branches, contact StreamRift."
    echo "══════════════════════════════════════════════════════"
    echo ""
    exit 1
  fi
done

exit 0
